// HumanFirst: Content Script for Google Search Enhancement

function applySettings() {
    chrome.storage.local.get(['hideAI', 'showBadges', 'safeYear'], (result) => {
        const canHide = result.hideAI === true; // Default off
        const canBadge = result.showBadges === true; // Default off
        const selectedYear = result.safeYear || "2022";

        if (canHide) {
            hideAIElements();
        }

        if (canBadge) {
            addHumanBadges();
        }

        injectHumanFirstButton(selectedYear);
    });
}

function hideAIElements() {
    const aiSelectors = [
        '[data-as="1"]',
        'div[data-entityid="AI_OVERVIEW"]',
        '.MljYf',
        '.f67HWc',
        '.yPsnN'
    ];
    aiSelectors.forEach(sel => {
        const el = document.querySelector(sel);
        if (el) el.style.display = 'none';
    });
}

function injectHumanFirstButton(year) {
    const target = document.querySelector('#extabar') || document.querySelector('#top_nav');
    if (!target || document.querySelector('.humanfirst-injected-btn')) return;

    const btn = document.createElement('button');
    btn.className = 'humanfirst-injected-btn';
    btn.innerHTML = '⏳ GO TO BEFORE 2022 (NO AI ERA) OLD SEARCH RESULTS';
    
    // Inline styles to ensure green color (overrides any caching issues)
    btn.style.cssText = `
        background: linear-gradient(135deg, #0a0f0a, #0f1a0f) !important;
        color: #86efac !important;
        border: 2px solid #22c55e !important;
        border-radius: 12px !important;
        padding: 12px 24px !important;
        margin: 15px 0 !important;
        font-weight: 700 !important;
        font-size: 14px !important;
        cursor: pointer !important;
        box-shadow: 0 0 20px rgba(34, 197, 94, 0.3), inset 0 0 15px rgba(34, 197, 94, 0.1) !important;
        display: flex !important;
        align-items: center !important;
        gap: 10px !important;
        transition: all 0.4s cubic-bezier(0.16, 1, 0.3, 1) !important;
        text-transform: uppercase !important;
        letter-spacing: 1px !important;
        z-index: 9999 !important;
        font-family: 'DM Sans', sans-serif !important;
    `;
    
    btn.onmouseover = () => {
        btn.style.background = 'linear-gradient(135deg, #14532d, #22c55e) !important';
        btn.style.color = '#ffffff !important';
        btn.style.transform = 'translateY(-3px)';
        btn.style.boxShadow = '0 0 30px rgba(34, 197, 94, 0.5), inset 0 0 20px rgba(34, 197, 94, 0.2) !important';
    };
    
    btn.onmouseout = () => {
        btn.style.background = 'linear-gradient(135deg, #0a0f0a, #0f1a0f) !important';
        btn.style.color = '#86efac !important';
        btn.style.transform = 'translateY(0)';
        btn.style.boxShadow = '0 0 20px rgba(34, 197, 94, 0.3), inset 0 0 15px rgba(34, 197, 94, 0.1) !important';
    };
    
    btn.onclick = () => {
        const queryParams = new URLSearchParams(window.location.search);
        const query = queryParams.get('q');
        if (query) {
            const dateStr = (year === "2022") ? "11/30/2022" : `12/31/${year}`;
            window.location.href = `https://www.google.com/search?q=${encodeURIComponent(query)}&tbs=cdr:1,cd_max:${dateStr}`;
        }
    };

    target.prepend(btn);
}

function addHumanBadges() {
    const results = document.querySelectorAll('.g, .tF2Cxc');

    results.forEach(result => {
        const snippet = result.innerText;
        const dateMatch = snippet.match(/(?:Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\s+\d{1,2},\s+\d{4}/i);

        if (dateMatch) {
            const date = new Date(dateMatch[0]);
            const cutoff = new Date('2022-11-30');

            if (date <= cutoff && !result.querySelector('.humanfirst-badge')) {
                const titleLink = result.querySelector('h3');
                if (titleLink) {
                    const badge = document.createElement('span');
                    badge.className = 'humanfirst-badge';
                    badge.innerHTML = '★ Verified Human Era';
                    
                    // Inline styles to ensure proper display (not inverted)
                    badge.style.cssText = `
                        display: inline-flex !important;
                        align-items: center !important;
                        background: rgba(34, 197, 94, 0.15) !important;
                        backdrop-filter: blur(5px) !important;
                        -webkit-backdrop-filter: blur(5px) !important;
                        color: #22c55e !important;
                        border: 1px solid rgba(34, 197, 94, 0.5) !important;
                        font-size: 11px !important;
                        font-weight: 800 !important;
                        padding: 4px 10px !important;
                        border-radius: 6px !important;
                        margin-left: 12px !important;
                        text-transform: uppercase !important;
                        box-shadow: 0 2px 8px rgba(34, 197, 94, 0.2) !important;
                        font-family: 'DM Sans', sans-serif !important;
                        letter-spacing: 0.5px !important;
                    `;
                    
                    titleLink.after(badge);
                }
            }
        }
    });
}

// HumanFirst: Safe Mutation Observer
const observer = new MutationObserver(() => applySettings());

function startObserving() {
    const targetNode = document.body || document.documentElement;
    if (targetNode) {
        observer.observe(targetNode, { childList: true, subtree: true });
        applySettings();
    } else {
        setTimeout(startObserving, 50);
    }
}

if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', startObserving);
} else {
    startObserving();
}

applySettings();
